"""Security helpers for the Flask application."""

from __future__ import annotations

import os
from functools import wraps
from typing import Callable, Iterable, Optional
from urllib.parse import urlparse

from flask import current_app, jsonify, request


def _load_api_keys() -> Iterable[str]:
    keys = os.getenv("API_KEYS")
    if keys:
        return [key.strip() for key in keys.split(",") if key.strip()]

    fallback = os.getenv("API_KEY")
    return [fallback] if fallback else []


def require_api_key(view_func: Callable) -> Callable:
    """Protect a view with an API key requirement."""

    @wraps(view_func)
    def wrapper(*args, **kwargs):
        provided_key = request.headers.get("X-API-Key")
        valid_keys = _load_api_keys()

        if not valid_keys:
            current_app.logger.warning(
                "API key protection is enabled but no API keys are configured."
            )
            return (
                jsonify(
                    {
                        "status": "error",
                        "message": "API key authentication is not configured.",
                    }
                ),
                503,
            )

        if provided_key not in valid_keys:
            return (
                jsonify({"status": "error", "message": "Unauthorized"}),
                401,
            )

        return view_func(*args, **kwargs)

    return wrapper


def validate_external_url(url: str) -> Optional[str]:
    """Validate an external URL before issuing an HTTP request."""

    if not url:
        return "URL is required"

    parsed = urlparse(url)

    if parsed.scheme not in {"http", "https"}:
        return "Only HTTP and HTTPS URLs are allowed"

    allowed_hosts = [
        host.strip().lower()
        for host in os.getenv("ALLOWED_DOWNLOAD_HOSTS", "").split(",")
        if host.strip()
    ]

    if not allowed_hosts:
        return (
            "Downloading from external URLs is disabled. Configure ALLOWED_DOWNLOAD_HOSTS."
        )

    hostname = (parsed.hostname or "").lower()

    if hostname not in allowed_hosts:
        return f"Host '{hostname}' is not allowed"

    return None


__all__ = ["require_api_key", "validate_external_url"]

