"""State management utilities for multi-worker deployments.

This module provides small helper abstractions for persisting ephemeral
application state such as cached file metadata and background task snapshots.
The previous implementation relied on in-memory dictionaries which caused
inconsistencies when the application was served by multiple Gunicorn workers.
By using a lightweight SQLite backend we ensure that all workers share a
single source of truth without introducing heavy external dependencies.  The
SQLite database lives alongside the application files and is safe for
multi-process access thanks to SQLite's built-in locking.
"""

from __future__ import annotations

import json
import os
import sqlite3
import threading
import time
from typing import Any, Dict, Optional


STATE_DB_PATH = os.getenv(
    "STATE_DB_PATH",
    os.path.join(os.path.dirname(os.path.dirname(__file__)), "runtime_state.sqlite3"),
)


class SQLiteStateStore:
    """Persist simple dictionaries in SQLite for cross-worker visibility."""

    def __init__(self, db_path: str) -> None:
        self._db_path = db_path
        self._lock = threading.Lock()
        self._ensure_directories()
        self._initialize()

    # ------------------------------------------------------------------
    # Internal helpers
    # ------------------------------------------------------------------
    def _ensure_directories(self) -> None:
        directory = os.path.dirname(self._db_path)
        if directory and not os.path.exists(directory):
            os.makedirs(directory, exist_ok=True)

    def _connection(self) -> sqlite3.Connection:
        return sqlite3.connect(self._db_path, timeout=30, check_same_thread=False)

    def _initialize(self) -> None:
        with self._connection() as conn:
            conn.execute(
                """
                CREATE TABLE IF NOT EXISTS file_cache (
                    cache_key TEXT PRIMARY KEY,
                    payload TEXT NOT NULL,
                    expires_at REAL
                )
                """
            )
            conn.execute(
                """
                CREATE TABLE IF NOT EXISTS background_tasks (
                    task_id TEXT PRIMARY KEY,
                    payload TEXT NOT NULL,
                    updated_at REAL NOT NULL
                )
                """
            )

    # ------------------------------------------------------------------
    # File cache helpers
    # ------------------------------------------------------------------
    def get_cache(self, key: str) -> Optional[Dict[str, Any]]:
        with self._lock, self._connection() as conn:
            row = conn.execute(
                "SELECT payload, expires_at FROM file_cache WHERE cache_key = ?",
                (key,),
            ).fetchone()

        if not row:
            return None

        payload_raw, expires_at = row
        if expires_at and expires_at < time.time():
            self.delete_cache(key)
            return None

        try:
            return json.loads(payload_raw)
        except json.JSONDecodeError:
            self.delete_cache(key)
            return None

    def set_cache(self, key: str, value: Dict[str, Any], ttl: int) -> None:
        expires_at = time.time() + ttl if ttl else None
        payload_raw = json.dumps(value)
        with self._lock, self._connection() as conn:
            conn.execute(
                "REPLACE INTO file_cache (cache_key, payload, expires_at) VALUES (?, ?, ?)",
                (key, payload_raw, expires_at),
            )

    def delete_cache(self, key: str) -> None:
        with self._lock, self._connection() as conn:
            conn.execute("DELETE FROM file_cache WHERE cache_key = ?", (key,))

    def cleanup_expired_cache(self) -> None:
        with self._lock, self._connection() as conn:
            conn.execute(
                "DELETE FROM file_cache WHERE expires_at IS NOT NULL AND expires_at < ?",
                (time.time(),),
            )

    # ------------------------------------------------------------------
    # Background task helpers
    # ------------------------------------------------------------------
    def get_task(self, task_id: str) -> Optional[Dict[str, Any]]:
        with self._lock, self._connection() as conn:
            row = conn.execute(
                "SELECT payload FROM background_tasks WHERE task_id = ?",
                (task_id,),
            ).fetchone()

        if not row:
            return None

        try:
            return json.loads(row[0])
        except json.JSONDecodeError:
            self.delete_task(task_id)
            return None

    def upsert_task(self, task_id: str, payload: Dict[str, Any]) -> None:
        serialized = json.dumps(payload)
        with self._lock, self._connection() as conn:
            conn.execute(
                "REPLACE INTO background_tasks (task_id, payload, updated_at) VALUES (?, ?, ?)",
                (task_id, serialized, time.time()),
            )

    def delete_task(self, task_id: str) -> None:
        with self._lock, self._connection() as conn:
            conn.execute("DELETE FROM background_tasks WHERE task_id = ?", (task_id,))

    def cleanup_stale_tasks(self, retention_seconds: int) -> None:
        cutoff = time.time() - retention_seconds
        with self._lock, self._connection() as conn:
            conn.execute(
                "DELETE FROM background_tasks WHERE updated_at < ?",
                (cutoff,),
            )


state_store = SQLiteStateStore(STATE_DB_PATH)


def perform_housekeeping(retention_seconds: int) -> None:
    """Remove expired cache entries and stale task snapshots."""

    state_store.cleanup_expired_cache()
    state_store.cleanup_stale_tasks(retention_seconds)


__all__ = [
    "state_store",
    "perform_housekeeping",
]

