"""Gunicorn configuration for running the application in production."""

from __future__ import annotations

import multiprocessing
import os
from typing import Any

# Bind to the address provided via environment variables or fall back to a
# production friendly default.
bind = os.getenv("GUNICORN_BIND", "0.0.0.0:8000")

# Use threaded workers to remain compatible with Flask's threaded assumptions
# while still benefitting from Gunicorn's worker management.
worker_class = "gthread"
threads = int(os.getenv("GUNICORN_THREADS", "8"))
workers = int(
    os.getenv(
        "GUNICORN_WORKERS",
        str((multiprocessing.cpu_count() * 2) + 1),
    )
)

# General hardening and performance related options.
preload_app = True
max_requests = int(os.getenv("GUNICORN_MAX_REQUESTS", "1000"))
max_requests_jitter = int(os.getenv("GUNICORN_MAX_REQUESTS_JITTER", "50"))
keepalive = int(os.getenv("GUNICORN_KEEPALIVE", "5"))
limit_request_line = int(os.getenv("GUNICORN_LIMIT_REQUEST_LINE", "4094"))
limit_request_fields = int(os.getenv("GUNICORN_LIMIT_REQUEST_FIELDS", "100"))
limit_request_field_size = int(os.getenv("GUNICORN_LIMIT_REQUEST_FIELD_SIZE", "8190"))

# Timeouts tuned for production deployments behind a reverse proxy.
timeout = int(os.getenv("GUNICORN_TIMEOUT", "30"))
graceful_timeout = int(os.getenv("GUNICORN_GRACEFUL_TIMEOUT", "30"))

# Logging configuration defaults to JSON structured production logging but can
# be overridden via the environment.
loglevel = os.getenv("GUNICORN_LOG_LEVEL", "info")
accesslog = os.getenv("GUNICORN_ACCESS_LOG", "-")
errorlog = os.getenv("GUNICORN_ERROR_LOG", "-")
capture_output = True

# Respect proxy forwarded headers for schemes so that Flask knows when it is
# serving secure traffic.
secure_scheme_headers = {"X-FORWARDED-PROTO": "https"}
forwarded_allow_ips = os.getenv("GUNICORN_FORWARDED_ALLOW_IPS", "127.0.0.1")


def when_ready(server: Any) -> None:
    """Log an informational message when the master process is ready."""

    server.log.info("Gunicorn production configuration loaded (gthread workers)")


def post_worker_init(worker: Any) -> None:
    """Initialise structured logging for each worker process."""

    try:
        from production_logging import configure_production_logging

        configure_production_logging()
    except Exception as exc:  # pragma: no cover - defensive logging
        worker.log.exception("Failed to configure production logging: %s", exc)


raw_env = ["FLASK_ENV=production"]
