"""Production logging helpers with JSON structured output and optional Sentry support."""

from __future__ import annotations

import json
import logging
import os
from logging.handlers import RotatingFileHandler
from pathlib import Path
from typing import Any, Dict, Optional

try:  # pragma: no cover - optional dependency
    import sentry_sdk
    from sentry_sdk.integrations.logging import LoggingIntegration
except ImportError:  # pragma: no cover - sentry is optional
    sentry_sdk = None  # type: ignore
    LoggingIntegration = None  # type: ignore


DEFAULT_LOG_RECORD_KEYS = {
    "name",
    "msg",
    "args",
    "levelname",
    "levelno",
    "pathname",
    "filename",
    "module",
    "exc_info",
    "exc_text",
    "stack_info",
    "lineno",
    "funcName",
    "created",
    "msecs",
    "relativeCreated",
    "thread",
    "threadName",
    "processName",
    "process",
    "message",
}


class JsonFormatter(logging.Formatter):
    """A logging formatter that emits structured JSON lines."""

    def format(self, record: logging.LogRecord) -> str:
        log_record: Dict[str, Any] = {
            "timestamp": self.formatTime(record, self.datefmt),
            "level": record.levelname,
            "logger": record.name,
            "message": record.getMessage(),
            "pathname": record.pathname,
            "lineno": record.lineno,
        }

        if record.exc_info:
            log_record["exc_info"] = self.formatException(record.exc_info)
        if record.stack_info:
            log_record["stack_info"] = record.stack_info

        for key, value in record.__dict__.items():
            if key in DEFAULT_LOG_RECORD_KEYS or key.startswith("_"):
                continue
            try:
                json.dumps(value)
                log_record[key] = value
            except TypeError:
                log_record[key] = str(value)

        return json.dumps(log_record, ensure_ascii=False)


def _create_rotating_handler(
    log_file: Path,
    level: int,
    max_bytes: int,
    backup_count: int,
    formatter: logging.Formatter,
) -> RotatingFileHandler:
    handler = RotatingFileHandler(
        log_file,
        maxBytes=max_bytes,
        backupCount=backup_count,
        encoding="utf-8",
    )
    handler.setLevel(level)
    handler.setFormatter(formatter)
    return handler


def configure_production_logging(
    app=None,
    *,
    log_dir: Optional[str] = None,
    log_level: Optional[str] = None,
) -> None:
    """Configure JSON structured logging for production deployments."""

    # Resolve configuration from environment if not provided explicitly
    effective_log_dir = log_dir or os.getenv("PRODUCTION_LOG_DIR", "logs")
    level_name = (log_level or os.getenv("LOG_LEVEL", "INFO")).upper()

    level = getattr(logging, level_name, logging.INFO)
    formatter = JsonFormatter(datefmt="%Y-%m-%dT%H:%M:%S%z")

    log_path: Optional[Path] = None
    try:
        log_path = Path(effective_log_dir)
        log_path.mkdir(parents=True, exist_ok=True)
    except OSError:
        log_path = None

    root_logger = logging.getLogger()
    root_logger.setLevel(level)

    # Remove existing handlers to avoid duplicate messages
    for handler in list(root_logger.handlers):
        root_logger.removeHandler(handler)

    console_handler = logging.StreamHandler()
    console_handler.setLevel(level)
    console_handler.setFormatter(formatter)
    root_logger.addHandler(console_handler)

    file_handler: Optional[RotatingFileHandler] = None
    if log_path is not None:
        log_file = log_path / os.getenv("PRODUCTION_LOG_FILE", "app.log")
        max_bytes = int(os.getenv("LOG_FILE_MAX_BYTES", 10 * 1024 * 1024))
        backup_count = int(os.getenv("LOG_FILE_BACKUP_COUNT", 5))
        file_handler = _create_rotating_handler(
            log_file,
            level,
            max_bytes,
            backup_count,
            formatter,
        )
        root_logger.addHandler(file_handler)
    else:
        root_logger.warning("Falling back to console-only logging; could not access %s", effective_log_dir)

    if app is not None:
        app.logger.handlers = []
        app.logger.setLevel(level)
        if file_handler is not None:
            app.logger.addHandler(file_handler)
        app.logger.addHandler(console_handler)

    _configure_sentry(level)


def _configure_sentry(level: int) -> None:
    """Initialize Sentry logging integration if configured via environment."""

    dsn = os.getenv("SENTRY_DSN")
    if not dsn:
        return

    if sentry_sdk is None or LoggingIntegration is None:
        logging.getLogger(__name__).warning(
            "SENTRY_DSN provided but sentry-sdk is not installed; skipping Sentry setup."
        )
        return

    if sentry_sdk.Hub.current.client is not None:
        return

    traces_sample_rate = float(os.getenv("SENTRY_TRACES_SAMPLE_RATE", "0.0"))
    send_pii = os.getenv("SENTRY_SEND_PII", "false").lower() in {"1", "true", "yes"}

    sentry_logging = LoggingIntegration(
        level=level,
        event_level=logging.ERROR,
    )

    sentry_sdk.init(
        dsn=dsn,
        integrations=[sentry_logging],
        traces_sample_rate=traces_sample_rate,
        send_default_pii=send_pii,
        environment=os.getenv("SENTRY_ENVIRONMENT", "production"),
    )
