#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
import sys
import signal
import logging
from pathlib import Path

# اضافه کردن دایرکتوری parent به path
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

def check_requirements():
    """
    بررسی وجود کتابخانه‌های مورد نیاز
    """
    required_packages = {
        'flask': 'Flask',
        'flask_cors': 'flask-cors',
        'openpyxl': 'openpyxl',
        'pdfplumber': 'pdfplumber',
        'pandas': 'pandas',
        'requests': 'requests'
    }
    
    missing_packages = []
    
    for package, pip_name in required_packages.items():
        try:
            __import__(package)
        except ImportError:
            missing_packages.append(pip_name)
    
    if missing_packages:
        print("Error: Missing required packages:")
        for package in missing_packages:
            print(f"  - {package}")
        print("\nInstall them using:")
        print(f"  pip install {' '.join(missing_packages)}")
        sys.exit(1)

def create_env_file():
    """
    ایجاد فایل .env نمونه اگر وجود نداشته باشد
    """
    env_file = Path('.env')
    env_example = Path('.env.example')
    
    if not env_file.exists() and not env_example.exists():
        example_content = """# Flask Configuration
FLASK_ENV=development
FLASK_DEBUG=True
SECRET_KEY=your-secret-key-here

# API Configuration
CRM_API_URL=https://your-crm-api.com/update_status
API_TIMEOUT=30

# CORS Configuration
ALLOWED_ORIGINS=*

# File Upload Configuration
MAX_CONTENT_LENGTH=16777216  # 16MB in bytes

# Logging Configuration
LOG_LEVEL=INFO
LOG_FILE_MAX_BYTES=10485760  # 10MB
LOG_FILE_BACKUP_COUNT=5
"""
        try:
            env_example.write_text(example_content)
            print(f"Created {env_example} file. Please configure your settings.")
        except Exception as e:
            print(f"Warning: Could not create .env.example: {str(e)}")

def signal_handler(sig, frame):
    """
    مدیریت سیگنال‌های سیستم عامل
    """
    print('\n')
    logging.info('Shutting down gracefully...')
    sys.exit(0)

def main():
    """
    تابع اصلی برای اجرای برنامه
    """
    try:
        # بررسی requirements
        check_requirements()
        
        # ایجاد فایل env نمونه
        create_env_file()
        
        # تنظیم signal handler
        signal.signal(signal.SIGINT, signal_handler)
        signal.signal(signal.SIGTERM, signal_handler)
        
        # import و اجرای app
        from app import app
        
        # خواندن تنظیمات از محیط
        host = os.getenv('FLASK_HOST', '0.0.0.0')
        port = int(os.getenv('FLASK_PORT', 5000))
        debug = os.getenv('FLASK_DEBUG', 'True').lower() in ['true', '1', 'yes']
        
        # بررسی پورت
        import socket
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        result = sock.connect_ex((host, port))
        sock.close()
        
        if result == 0:
            print(f"Warning: Port {port} is already in use.")
            print("Trying alternative port...")
            
            # جستجوی پورت خالی
            for alt_port in range(port + 1, port + 100):
                sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                result = sock.connect_ex((host, alt_port))
                sock.close()
                if result != 0:
                    port = alt_port
                    print(f"Using port {port}")
                    break
        
        # نمایش اطلاعات شروع
        print("=" * 50)
        print("Flask Transaction Checker API")
        print("=" * 50)
        print(f"Host: {host}")
        print(f"Port: {port}")
        print(f"Debug: {debug}")
        print(f"URL: http://{host if host != '0.0.0.0' else 'localhost'}:{port}")
        print("=" * 50)
        print("Press CTRL+C to stop the server")
        print("")
        
        # اجرای سرور
        app.run(
            host=host,
            port=port,
            debug=debug,
            use_reloader=debug,
            threaded=True  # فعال کردن threading برای پردازش همزمان
        )
        
    except ImportError as e:
        print(f"Error: Failed to import required modules: {str(e)}")
        print("\nMake sure all dependencies are installed:")
        print("  pip install -r requirements.txt")
        sys.exit(1)
        
    except Exception as e:
        print(f"Critical error: {str(e)}")
        logging.error(f"Failed to start application: {str(e)}", exc_info=True)
        sys.exit(1)

if __name__ == "__main__":
    main()