from flask import Flask
from flask_cors import CORS
import logging
import os
import sys
from logging.handlers import RotatingFileHandler
from datetime import datetime

# ایجاد Flask app
app = Flask(__name__)

# تنظیمات CORS با مدیریت خطا
try:
    CORS(app, resources={
        r"/*": {
            "origins": os.getenv("ALLOWED_ORIGINS", "*").split(","),
            "methods": ["GET", "POST", "OPTIONS"],
            "allow_headers": ["Content-Type", "Authorization"]
        }
    })
except Exception as e:
    print(f"Warning: CORS configuration failed: {str(e)}")
    CORS(app)  # fallback به تنظیمات پیش‌فرض

# تنظیمات محیطی
app.config['MAX_CONTENT_LENGTH'] = 16 * 1024 * 1024  # 16MB max file size
app.config['JSON_AS_ASCII'] = False  # پشتیبانی از UTF-8
app.config['JSON_SORT_KEYS'] = False

def setup_logging():
    """
    پیکربندی سیستم لاگ‌گیری با مدیریت خطا
    """
    try:
        # ایجاد پوشه logs
        log_dir = 'logs'
        if not os.path.exists(log_dir):
            try:
                os.makedirs(log_dir)
            except PermissionError:
                # اگر دسترسی نداریم، از /tmp استفاده کنیم
                log_dir = '/tmp/flask_logs'
                os.makedirs(log_dir, exist_ok=True)
        
        # بررسی دسترسی نوشتن
        test_file = os.path.join(log_dir, '.test_write')
        try:
            with open(test_file, 'w') as f:
                f.write('test')
            os.remove(test_file)
        except (IOError, PermissionError):
            print(f"Warning: Cannot write to {log_dir}, using console logging only")
            log_dir = None
        
        # تنظیم فرمت لاگ
        log_format = logging.Formatter(
            '%(asctime)s - %(name)s - %(levelname)s - %(message)s',
            datefmt='%Y-%m-%d %H:%M:%S'
        )
        
        # تنظیم لاگر اصلی
        root_logger = logging.getLogger()
        root_logger.setLevel(logging.INFO)
        
        # حذف handler های قبلی
        for handler in root_logger.handlers[:]:
            root_logger.removeHandler(handler)
        
        # Console handler (همیشه فعال)
        console_handler = logging.StreamHandler(sys.stdout)
        console_handler.setLevel(logging.INFO)
        console_handler.setFormatter(log_format)
        root_logger.addHandler(console_handler)
        
        if log_dir:
            # Rotating file handler برای لاگ‌های عمومی
            try:
                info_handler = RotatingFileHandler(
                    os.path.join(log_dir, 'app.log'),
                    maxBytes=10 * 1024 * 1024,  # 10MB
                    backupCount=5,
                    encoding='utf-8'
                )
                info_handler.setLevel(logging.INFO)
                info_handler.setFormatter(log_format)
                root_logger.addHandler(info_handler)
                
            except Exception as e:
                print(f"Warning: Could not create info log handler: {str(e)}")
            
            # Rotating file handler برای لاگ‌های خطا
            try:
                error_handler = RotatingFileHandler(
                    os.path.join(log_dir, 'error.log'),
                    maxBytes=10 * 1024 * 1024,  # 10MB
                    backupCount=5,
                    encoding='utf-8'
                )
                error_handler.setLevel(logging.ERROR)
                error_handler.setFormatter(log_format)
                root_logger.addHandler(error_handler)
                
            except Exception as e:
                print(f"Warning: Could not create error log handler: {str(e)}")
            
            # لاگ‌های دسترسی برای Flask
            try:
                access_handler = RotatingFileHandler(
                    os.path.join(log_dir, 'access.log'),
                    maxBytes=10 * 1024 * 1024,  # 10MB
                    backupCount=5,
                    encoding='utf-8'
                )
                access_handler.setLevel(logging.INFO)
                access_handler.setFormatter(log_format)
                
                # اضافه کردن به لاگر werkzeug (Flask's internal logger)
                werkzeug_logger = logging.getLogger('werkzeug')
                werkzeug_logger.addHandler(access_handler)
                
            except Exception as e:
                print(f"Warning: Could not create access log handler: {str(e)}")
        
        # لاگر برای ماژول app
        app.logger.setLevel(logging.INFO)
        
        # لاگ اولیه
        app.logger.info("=" * 50)
        app.logger.info(f"Flask application starting at {datetime.now()}")
        app.logger.info(f"Log directory: {log_dir if log_dir else 'Console only'}")
        app.logger.info(f"Python version: {sys.version}")
        app.logger.info("=" * 50)
        
        return True
        
    except Exception as e:
        print(f"Critical: Logging setup failed: {str(e)}")
        # حداقل تنظیمات لاگینگ
        logging.basicConfig(
            level=logging.INFO,
            format='%(asctime)s - %(levelname)s - %(message)s',
            stream=sys.stdout
        )
        return False

def create_required_directories():
    """
    ایجاد دایرکتوری‌های مورد نیاز
    """
    directories = ['logs', 'outputs', 'backups', 'temp']
    
    for directory in directories:
        try:
            if not os.path.exists(directory):
                os.makedirs(directory)
                app.logger.info(f"Created directory: {directory}")
            else:
                # بررسی دسترسی
                if not os.access(directory, os.W_OK):
                    app.logger.warning(f"No write permission for directory: {directory}")
        except Exception as e:
            app.logger.error(f"Could not create directory {directory}: {str(e)}")

# Error handlers
@app.errorhandler(404)
def not_found(error):
    """مدیریت خطای 404"""
    return {
        "status": "error",
        "message": "Endpoint not found",
        "error_code": 404
    }, 404

@app.errorhandler(405)
def method_not_allowed(error):
    """مدیریت خطای 405"""
    return {
        "status": "error",
        "message": "Method not allowed",
        "error_code": 405
    }, 405

@app.errorhandler(413)
def request_entity_too_large(error):
    """مدیریت خطای حجم زیاد فایل"""
    return {
        "status": "error",
        "message": "File size exceeds maximum limit (16MB)",
        "error_code": 413
    }, 413

@app.errorhandler(500)
def internal_error(error):
    """مدیریت خطای 500"""
    app.logger.error(f"Internal server error: {str(error)}")
    return {
        "status": "error",
        "message": "Internal server error occurred",
        "error_code": 500
    }, 500

@app.errorhandler(Exception)
def handle_exception(error):
    """مدیریت خطاهای غیرمنتظره"""
    app.logger.error(f"Unhandled exception: {str(error)}", exc_info=True)
    return {
        "status": "error",
        "message": "An unexpected error occurred",
        "error_code": 500,
        "details": str(error) if app.debug else None
    }, 500

# Middleware برای لاگ کردن درخواست‌ها
@app.before_request
def log_request():
    """لاگ کردن اطلاعات درخواست"""
    try:
        app.logger.info(f"Request: {request.method} {request.path} from {request.remote_addr}")
        
        # لاگ کردن payload در حالت debug
        if app.debug and request.is_json:
            try:
                data = request.get_json(silent=True)
                if data:
                    # محدود کردن طول لاگ
                    data_str = str(data)[:500]
                    app.logger.debug(f"Request payload: {data_str}")
            except:
                pass
    except Exception as e:
        app.logger.error(f"Error in before_request: {str(e)}")

@app.after_request
def log_response(response):
    """لاگ کردن اطلاعات پاسخ"""
    try:
        app.logger.info(f"Response: {response.status_code} for {request.method} {request.path}")
    except Exception as e:
        app.logger.error(f"Error in after_request: {str(e)}")
    return response

# اجرای setup
try:
    setup_logging()
    create_required_directories()
    app.logger.info("Flask application initialized successfully")
except Exception as e:
    print(f"Error during initialization: {str(e)}")
    # ادامه با حداقل تنظیمات
    logging.basicConfig(level=logging.INFO)

# Import routes در آخر برای جلوگیری از circular import
try:
    from . import routes
    app.logger.info("Routes imported successfully")
except ImportError as e:
    app.logger.error(f"Failed to import routes: {str(e)}")
    # ایجاد یک route ساده برای health check
    @app.route('/')
    def emergency_health_check():
        return {"status": "partial", "message": "Routes not loaded", "error": str(e)}, 503