import json
from flask import request, jsonify
from app import app
from .utils import highlight_row, green_fill, red_fill
import openpyxl
import os
import pdfplumber
import pandas as pd
import requests
import re
from datetime import datetime
import io
import traceback
from typing import Optional, Dict, Any, List, Tuple

# تنظیمات پیش‌فرض برای timeout
DEFAULT_REQUEST_TIMEOUT = 30  # ثانیه
MAX_RETRIES = 3

class FileProcessingError(Exception):
    """خطای سفارشی برای پردازش فایل‌ها"""
    pass

class APIConnectionError(Exception):
    """خطای سفارشی برای ارتباط با API"""
    pass

def safe_get_json() -> Optional[Dict[str, Any]]:
    """
    دریافت امن JSON از request با مدیریت خطا
    """
    try:
        if not request.is_json:
            return None
        return request.get_json(force=True, silent=True)
    except Exception as e:
        app.logger.error(f"Error parsing JSON from request: {str(e)}")
        return None

def safe_get_value(data: Dict[str, Any], key: str, default=None, data_type=None):
    """
    دریافت امن مقدار از dictionary با type checking
    """
    try:
        value = data.get(key, default)
        if value is None:
            return default
        
        if data_type:
            if data_type == float:
                # تبدیل به float با حذف کاما
                return float(str(value).replace(",", "").strip())
            elif data_type == str:
                return str(value).strip()
            elif data_type == int:
                return int(value)
            elif data_type == list:
                return value if isinstance(value, list) else [value]
        return value
    except (ValueError, TypeError, AttributeError) as e:
        app.logger.warning(f"Error converting {key} to {data_type}: {str(e)}")
        return default

def retry_request(func, *args, max_retries=MAX_RETRIES, **kwargs):
    """
    اجرای تابع با retry در صورت خطا
    """
    last_error = None
    for attempt in range(max_retries):
        try:
            return func(*args, **kwargs)
        except (requests.ConnectionError, requests.Timeout) as e:
            last_error = e
            app.logger.warning(f"Attempt {attempt + 1} failed: {str(e)}")
            if attempt < max_retries - 1:
                import time
                time.sleep(2 ** attempt)  # Exponential backoff
    raise APIConnectionError(f"Failed after {max_retries} attempts: {str(last_error)}")

def get_unique_filename(directory: str, base_name: str) -> str:
    """
    تولید نام یکتا برای فایل در یک دایرکتوری
    """
    name, ext = os.path.splitext(base_name)
    counter = 1
    new_name = base_name
    while os.path.exists(os.path.join(directory, new_name)):
        new_name = f"{name}_{counter}{ext}"
        counter += 1
    return new_name

def download_file(url: str, save_path: str) -> Optional[str]:
    """دانلود فایل از API"""
    try:
        response = requests.get(url, timeout=60)
        response.raise_for_status()
        with open(save_path, "wb") as f:
            f.write(response.content)
        app.logger.info(f"Downloaded file from {url} to {save_path}")
        return save_path
    except Exception as e:
        app.logger.error(f"Failed to download file from {url}: {str(e)}")
        return None

def update_crm_status(record_id: str, status: str, color: str) -> bool:
    """
    به‌روزرسانی وضعیت در CRM با مدیریت خطا
    """
    try:
        # بررسی پارامترهای ورودی
        if not all([record_id, status, color]):
            app.logger.warning(f"Invalid CRM update parameters: record_id={record_id}, status={status}, color={color}")
            return False
        
        crm_endpoint = os.getenv("CRM_API_URL", "https://your-crm-api.com/update_status")
        
        # بررسی URL معتبر
        if not crm_endpoint or crm_endpoint == "https://your-crm-api.com/update_status":
            app.logger.info(f"CRM endpoint not configured, skipping update for record {record_id}")
            return False
        
        payload = {
            "record_id": record_id,
            "status": status,
            "color": color,
            "timestamp": datetime.now().isoformat()
        }
        
        # ارسال با retry
        def send_request():
            return requests.post(
                crm_endpoint, 
                json=payload,
                timeout=DEFAULT_REQUEST_TIMEOUT,
                headers={"Content-Type": "application/json"}
            )
        
        response = retry_request(send_request)
        response.raise_for_status()
        
        app.logger.info(f"CRM updated for record {record_id}: status={status}, color={color}")
        return True
        
    except APIConnectionError as e:
        app.logger.error(f"Connection error updating CRM for record {record_id}: {str(e)}")
        return False
    except requests.HTTPError as e:
        app.logger.error(f"HTTP error updating CRM for record {record_id}: {e.response.status_code} - {e.response.text}")
        return False
    except Exception as e:
        app.logger.error(f"Unexpected error updating CRM for record {record_id}: {str(e)}")
        return False

def save_excel_via_api(workbook, filename: str, api_endpoint: Optional[str] = None) -> Dict[str, Any]:
    """
    ذخیره فایل اکسل و ارسال آن به API خارجی با مدیریت خطا
    """
    result = {
        "filename": filename,
        "status": "pending",
        "saved_locally": False,
        "sent_to_api": False,
        "errors": []
    }
    
    try:
        # ذخیره در memory buffer
        excel_buffer = io.BytesIO()
        workbook.save(excel_buffer)
        excel_buffer.seek(0)
        
        # ذخیره محلی با مدیریت خطا
        try:
            output_dir = "outputs"
            if not os.path.exists(output_dir):
                os.makedirs(output_dir)

            # نام یکتا برای فایل
            unique_filename = get_unique_filename(output_dir, filename)
            local_path = os.path.join(output_dir, unique_filename)

            workbook.save(local_path)
            result["saved_locally"] = True
            result["local_path"] = local_path
            result["filename"] = unique_filename

            app.logger.info(f"Excel file saved locally: {local_path}")
        except (IOError, PermissionError) as e:
            error_msg = f"Failed to save locally: {str(e)}"
            result["errors"].append(error_msg)
            app.logger.error(error_msg)
        
        # ارسال به API اگر endpoint مشخص شده
        if api_endpoint and api_endpoint != "None":
            try:
                files = {
                    'file': (
                        filename, 
                        excel_buffer.getvalue(),
                        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
                    )
                }
                
                def send_file():
                    return requests.post(
                        api_endpoint, 
                        files=files,
                        timeout=DEFAULT_REQUEST_TIMEOUT
                    )
                
                response = retry_request(send_file)
                response.raise_for_status()
                
                result["sent_to_api"] = True
                
                # پردازش response
                if response.headers.get('content-type', '').startswith('application/json'):
                    result["api_response"] = response.json()
                else:
                    result["api_response"] = response.text[:500]  # محدود کردن طول
                    
                app.logger.info(f"Excel file sent to API: {filename}")
                
            except APIConnectionError as e:
                error_msg = f"API connection failed: {str(e)}"
                result["errors"].append(error_msg)
                app.logger.error(error_msg)
            except requests.HTTPError as e:
                error_msg = f"API HTTP error: {e.response.status_code}"
                result["errors"].append(error_msg)
                app.logger.error(error_msg)
            except Exception as e:
                error_msg = f"API error: {str(e)}"
                result["errors"].append(error_msg)
                app.logger.error(error_msg)
        
        # تعیین وضعیت نهایی
        if result["saved_locally"] or result["sent_to_api"]:
            result["status"] = "success"
        else:
            result["status"] = "error"
            
    except Exception as e:
        result["status"] = "error"
        result["errors"].append(f"Unexpected error: {str(e)}")
        app.logger.error(f"Unexpected error in save_excel_via_api: {str(e)}")
    
    return result

def safe_load_workbook(file_path: str) -> Optional[openpyxl.Workbook]:
    """
    بارگذاری امن فایل اکسل
    """
    try:
        if not os.path.exists(file_path):
            app.logger.warning(f"File not found: {file_path}")
            return None
        
        if not os.access(file_path, os.R_OK):
            app.logger.error(f"No read permission for file: {file_path}")
            return None
        
        wb = openpyxl.load_workbook(file_path, data_only=True)
        app.logger.info(f"Successfully loaded: {file_path}")
        return wb
        
    except openpyxl.utils.exceptions.InvalidFileException as e:
        app.logger.error(f"Invalid Excel file {file_path}: {str(e)}")
        return None
    except Exception as e:
        app.logger.error(f"Error loading {file_path}: {str(e)}")
        return None

def safe_process_pdf(pdf_path: str) -> Tuple[Optional[pd.DataFrame], Optional[openpyxl.Workbook]]:
    """
    پردازش امن فایل PDF
    """
    try:
        if not os.path.exists(pdf_path):
            app.logger.warning(f"PDF file not found: {pdf_path}")
            return None, None
        
        all_data = []
        
        with pdfplumber.open(pdf_path) as pdf:
            for page_num, page in enumerate(pdf.pages, start=1):
                try:
                    tables = page.extract_tables()
                    if tables:
                        app.logger.info(f"Page {page_num}: Found {len(tables)} tables")
                        for table in tables:
                            if table:  # بررسی اینکه table خالی نباشه
                                for row in table:
                                    if row:  # بررسی اینکه row خالی نباشه
                                        processed_row = [
                                            str(cell).strip() if cell is not None else ""
                                            for cell in row
                                        ]
                                        all_data.append(processed_row)
                except Exception as e:
                    app.logger.error(f"Error processing page {page_num}: {str(e)}")
                    continue
        
        if not all_data:
            app.logger.warning(f"No data extracted from PDF: {pdf_path}")
            return None, None
        
        # تبدیل به DataFrame
        pdf_data = pd.DataFrame(all_data)
        app.logger.info(f"PDF data shape: {pdf_data.shape}")
        
        # ذخیره به اکسل
        output_dir = "outputs"
        if not os.path.exists(output_dir):
            os.makedirs(output_dir)
            
        excel_path = os.path.join(output_dir, get_unique_filename(output_dir, "pdf_extracted.xlsx"))
        pdf_data.to_excel(excel_path, index=False, header=False)
        
        # بارگذاری مجدد به عنوان workbook
        wb_pdf = openpyxl.load_workbook(excel_path)
        
        return pdf_data, wb_pdf
        
    except Exception as e:
        app.logger.error(f"Error processing PDF {pdf_path}: {str(e)}")
        app.logger.error(traceback.format_exc())
        return None, None

def jsonify_error(message: str, status_code: int = 400, details: Dict = None) -> tuple:
    """
    ایجاد پاسخ خطای استاندارد
    """
    error_response = {
        "status": "error",
        "message": message,
        "timestamp": datetime.now().isoformat()
    }
    
    if details:
        error_response["details"] = details
    
    app.logger.error(f"Error response: {message}")
    return jsonify(error_response), status_code

@app.route('/')
def health_check():
    """Health check endpoint"""
    try:
        return jsonify({
            "status": "healthy",
            "message": "API is running",
            "timestamp": datetime.now().isoformat()
        })
    except Exception as e:
        return jsonify_error("Health check failed", 500, {"error": str(e)})

@app.route('/check', methods=['POST'])
def check_transaction():
    """
    بررسی تراکنش‌ها با پشتیبانی از JSON و form-data
    """
    try:
        data = None
        file_paths = {}

        # اگر ورودی JSON خالص بود
        if request.is_json:
            data = request.get_json(silent=True)

        # اگر form-data بود
        elif "data" in request.form:
            try:
                data = json.loads(request.form["data"])
            except json.JSONDecodeError:
                return jsonify_error("فرمت data معتبر نیست (JSON).")

            # مدیریت فایل‌های آپلودی
            upload_dir = "uploads"
            if not os.path.exists(upload_dir):
                os.makedirs(upload_dir)

            if "sheet1" in request.files:
                sheet1_file = request.files["sheet1"]
                path = os.path.join(upload_dir, sheet1_file.filename)
                sheet1_file.save(path)
                file_paths["sheet1"] = path

            if "sheet2" in request.files:
                sheet2_file = request.files["sheet2"]
                path = os.path.join(upload_dir, sheet2_file.filename)
                sheet2_file.save(path)
                file_paths["sheet2"] = path

            if "input_pdf" in request.files:
                pdf_file = request.files["input_pdf"]
                path = os.path.join(upload_dir, pdf_file.filename)
                pdf_file.save(path)
                file_paths["pdf_input"] = path

        # اگر هیچ دیتایی نیامد
        if not data:
            return jsonify_error("هیچ داده‌ای ارسال نشده است یا JSON معتبر نیست.")

        app.logger.info(f"Received request with {len(data)} keys")

        # دانلود فایل‌ها از API
        file_api = data.get("file_api", {})
        if "sheet1" in file_api:
            file_paths["sheet1"] = download_file(file_api["sheet1"], os.path.join(upload_dir, "sheet1.xlsx"))
        if "sheet2" in file_api:
            file_paths["sheet2"] = download_file(file_api["sheet2"], os.path.join(upload_dir, "sheet2.xlsx"))
        if "pdf_input" in file_api:
            file_paths["pdf_input"] = download_file(file_api["pdf_input"], os.path.join(upload_dir, "input.pdf"))

        # اضافه کردن مسیر فایل‌ها به data
        data["file_paths"] = {**file_paths, **data.get("file_paths", {})}

        # استخراج مسیرها
        file_paths = safe_get_value(data, 'file_paths', {}, dict)
        BASE_DIR = os.path.dirname(os.path.abspath(__file__))
        SHEET1_PATH = file_paths.get('sheet1', os.path.join(BASE_DIR, "sheet1.xlsx"))
        SHEET2_PATH = file_paths.get('sheet2', os.path.join(BASE_DIR, "sheet2.xlsx"))
        PDF_INPUT_PATH = file_paths.get('pdf_input', os.path.join(BASE_DIR, "input.pdf"))

        save_api_endpoint = safe_get_value(data, 'save_api_endpoint', None, str)
        records = safe_get_value(data, 'records', [], list)

        if not records:
            excluded_keys = {'file_paths', 'save_api_endpoint', 'file_api', 'records'}
            record_data = {k: v for k, v in data.items() if k not in excluded_keys}
            if record_data:
                records = [record_data]

        if not records:
            return jsonify_error("هیچ رکوردی برای پردازش یافت نشد.")

        # ===== پردازش مثل قبل =====
        wb1 = safe_load_workbook(SHEET1_PATH)
        wb2 = safe_load_workbook(SHEET2_PATH)
        pdf_data, wb_pdf = safe_process_pdf(PDF_INPUT_PATH)

        if not any([wb1, wb2, pdf_data]):
            return jsonify_error("هیچ منبع داده‌ای در دسترس نیست.", 500)

        results = []
        for idx, record in enumerate(records):
            try:
                result = process_single_record(record, idx, wb1, wb2, pdf_data, wb_pdf)
                results.append(result)
            except Exception as e:
                app.logger.error(f"Error processing record {idx}: {str(e)}")
                results.append({
                    "record_id": f"record_{idx}",
                    "status": "error",
                    "details": {"message": str(e)}
                })

        save_results = []
        if wb1:
            save_results.append(save_excel_via_api(wb1, "sheet1_updated.xlsx", save_api_endpoint))
        if wb2:
            save_results.append(save_excel_via_api(wb2, "sheet2_updated.xlsx", save_api_endpoint))
        if wb_pdf:
            save_results.append(save_excel_via_api(wb_pdf, "pdf_extracted_updated.xlsx", save_api_endpoint))

        return jsonify({
            "status": "success",
            "results": results,
            "save_results": save_results,
            "timestamp": datetime.now().isoformat()
        })

    except Exception as e:
        app.logger.error(f"Unexpected error: {str(e)}")
        return jsonify_error("خطای غیرمنتظره در پردازش درخواست", 500, {"error": str(e)})

def process_single_record(record: Dict, idx: int, wb1, wb2, pdf_data, wb_pdf) -> Dict:
    """
    پردازش یک رکورد با مدیریت خطا
    """
    record_id = safe_get_value(record, "record_id", f"record_{idx}", str)
    
    result = {
        "record_id": record_id,
        "status": "not_found",
        "details": {
            "sources": [],
            "match_count": 0,
            "message": "اطلاعات یافت نشد.",
            "debug_info": []
        }
    }
    
    try:
        # تعیین منبع
        source = determine_source(record)
        if not source:
            result["details"]["debug_info"].append("منبع قابل تشخیص نیست.")
            return result
        
        app.logger.info(f"Record {record_id}: Source determined as {source}")
        
        # بررسی الزامات
        if not validate_record_requirements(record, source, result):
            return result
        
        # استخراج قیمت
        price = safe_get_value(record, "price", None, float)
        if price is None:
            result["details"]["debug_info"].append("مبلغ معتبر یافت نشد.")
            return result
        
        app.logger.info(f"Record {record_id}: Price = {price}")
        
        # پردازش بر اساس منبع
        confirmed = False
        confirmed_sources = []
        match_count = 0
        partial_match = False
        debug_info = []
        
        # پردازش Sheet1
        if source in ["all", "sheet1"] and wb1:
            sheet1_result = process_sheet1(record, price, wb1, record_id)
            if sheet1_result:
                if sheet1_result["type"] == "full":
                    confirmed = True
                    confirmed_sources.append("sheet1")
                else:
                    partial_match = True
                    confirmed_sources.append("sheet1_partial")
                match_count += 1
                debug_info.extend(sheet1_result.get("debug", []))
        
        # پردازش Sheet2
        if source in ["all", "sheet2"] and wb2:
            sheet2_result = process_sheet2(record, price, wb2, record_id)
            if sheet2_result:
                if sheet2_result["type"] == "full":
                    confirmed = True
                    confirmed_sources.append("sheet2")
                else:
                    partial_match = True
                    confirmed_sources.append("sheet2_partial")
                match_count += 1
                debug_info.extend(sheet2_result.get("debug", []))
        
        # پردازش PDF
        if source in ["all", "pdf"] and pdf_data is not None and wb_pdf:
            pdf_result = process_pdf(record, price, pdf_data, wb_pdf, record_id)
            if pdf_result:
                if pdf_result["type"] == "full":
                    confirmed = True
                    confirmed_sources.append("pdf")
                else:
                    partial_match = True
                    confirmed_sources.append("pdf_partial")
                match_count += 1
                debug_info.extend(pdf_result.get("debug", []))
        
        # تعیین وضعیت نهایی
        if confirmed:
            status = "confirmed"
            color = "green"
            message = "تراکنش تأیید شد."
        elif partial_match:
            status = "partial_match"
            color = "yellow"
            message = "تطابق جزئی یافت شد."
        else:
            status = "not_found"
            color = "red"
            message = "اطلاعات یافت نشد."
        
        result["status"] = status
        result["details"]["sources"] = confirmed_sources
        result["details"]["match_count"] = match_count
        result["details"]["message"] = message
        result["details"]["debug_info"] = debug_info
        
        # به‌روزرسانی CRM
        result["crm_updated"] = update_crm_status(record_id, status, color)
        
    except Exception as e:
        app.logger.error(f"Error processing record {record_id}: {str(e)}")
        result["status"] = "error"
        result["details"]["message"] = f"خطا: {str(e)}"
    
    return result

def determine_source(record: Dict) -> Optional[str]:
    """تعیین منبع بر اساس فیلدهای موجود"""
    if "source" in record:
        return record["source"].lower().strip()
    
    if "transaction_code" in record:
        return "sheet1"
    elif "payment_number" in record:
        return "sheet2"
    elif "document_number" in record:
        return "pdf"
    
    return None

def validate_record_requirements(record: Dict, source: str, result: Dict) -> bool:
    """بررسی الزامات رکورد"""
    requirements = {
        "sheet1": "transaction_code",
        "sheet2": "payment_number",
        "pdf": "document_number"
    }
    
    if source in requirements:
        required_field = requirements[source]
        if required_field not in record:
            result["details"]["debug_info"].append(f"فیلد {required_field} الزامی است.")
            return False
    
    return True

def process_sheet1(record: Dict, price: float, wb1, record_id: str) -> Optional[Dict]:
    """پردازش Sheet1 بر اساس کد تراکنش + مبلغ"""
    try:
        transaction_code = safe_get_value(record, "transaction_code", "", str)
        ws1 = wb1.active

        # هدرها در ردیف دوم هستن
        headers = [str(cell.value).strip() if cell.value else "" for cell in next(ws1.iter_rows(min_row=2, max_row=2))]
        app.logger.info(f"Sheet1 headers: {headers}")

        # پیدا کردن ستون‌ها
        col_transaction = headers.index("کد تراکنش")
        col_price = headers.index("مبلغ")

        # داده‌ها از ردیف سوم به بعد شروع میشن
        for row in ws1.iter_rows(min_row=3):
            try:
                cell_transaction = str(row[col_transaction].value).strip() if row[col_transaction].value else ""
                cell_price = str(row[col_price].value).replace(",", "").strip() if row[col_price].value else ""

                if not cell_transaction or not cell_price:
                    continue

                cell_price_float = float(cell_price)

                is_transaction_match = cell_transaction.lower() == transaction_code.lower()
                is_price_match = abs(cell_price_float - price) < 0.01

                if is_transaction_match and is_price_match:
                    highlight_row(ws1, row[0].row, green_fill)
                    app.logger.info(f"Sheet1 FULL match at row {row[0].row}: transaction={cell_transaction}, price={cell_price_float}")
                    return {"type": "full", "row": row[0].row, "debug": [f"sheet1_full row={row[0].row}"]}
                elif is_transaction_match or is_price_match:
                    highlight_row(ws1, row[0].row, red_fill)
                    app.logger.info(f"Sheet1 PARTIAL match at row {row[0].row}: transaction={cell_transaction}, price={cell_price_float}")
                    return {"type": "partial", "row": row[0].row, "debug": [f"sheet1_partial row={row[0].row}"]}

            except Exception as e:
                app.logger.error(f"Row error in Sheet1 (row {row[0].row}): {str(e)}")
                continue

    except Exception as e:
        app.logger.error(f"Error in process_sheet1: {str(e)}")

    return None

def process_sheet2(record: Dict, price: float, wb2, record_id: str) -> Optional[Dict]:
    """پردازش Sheet2 بر اساس شماره پرداخت + مبلغ (ترکیبی: هدر یا ستون ثابت)"""
    try:
        payment_number = safe_get_value(record, "payment_number", "", str)
        ws2 = wb2.active

        # نرمالایز کردن متن (حذف فاصله و نیم‌فاصله)
        def normalize(text):
            return str(text).strip().replace("‌", "").replace(" ", "")

        # تلاش اول: پیدا کردن هدرها
        headers = [str(cell.value).strip() if cell.value else "" for cell in next(ws2.iter_rows(min_row=2, max_row=2))]
        app.logger.info(f"Sheet2 headers (raw): {headers}")

        col_payment = next((i for i, h in enumerate(headers) if normalize(h) in ["شمارهپرداخت"]), None)
        col_price = next((i for i, h in enumerate(headers) if normalize(h) in ["مبلغ"]), None)

        # اگر هدر پیدا نشد، برو سراغ ستون‌های ثابت
        if col_payment is None or col_price is None:
            app.logger.warning("Headers not found in Sheet2, falling back to fixed columns (price=8, payment=9).")
            col_price, col_payment = 8, 9
            data_start_row = 2  # چون تو این حالت از ردیف دوم داده‌ها شروع میشه
        else:
            data_start_row = 3  # وقتی هدر هست، داده‌ها از ردیف سوم شروع میشن

        # داده‌ها از جایی که مشخص شد شروع میشن
        for row in ws2.iter_rows(min_row=data_start_row):
            try:
                cell_payment = str(row[col_payment].value).strip() if row[col_payment].value else ""
                cell_price = str(row[col_price].value).replace(",", "").strip() if row[col_price].value else ""

                if not cell_payment or not cell_price:
                    continue

                cell_price_float = float(cell_price)

                is_payment_match = cell_payment == payment_number
                is_price_match = abs(cell_price_float - price) < 0.01

                if is_payment_match and is_price_match:
                    highlight_row(ws2, row[0].row, green_fill)
                    app.logger.info(f"Sheet2 FULL match at row {row[0].row}: payment={cell_payment}, price={cell_price_float}")
                    return {"type": "full", "row": row[0].row, "debug": [f"sheet2_full row={row[0].row}"]}
                elif is_payment_match or is_price_match:
                    highlight_row(ws2, row[0].row, red_fill)
                    app.logger.info(f"Sheet2 PARTIAL match at row {row[0].row}: payment={cell_payment}, price={cell_price_float}")
                    return {"type": "partial", "row": row[0].row, "debug": [f"sheet2_partial row={row[0].row}"]}

            except Exception as e:
                app.logger.error(f"Row error in Sheet2 (row {row[0].row}): {str(e)}")
                continue

    except Exception as e:
        app.logger.error(f"Error in process_sheet2: {str(e)}")

    return None

def process_pdf(record: Dict, price: float, pdf_data: pd.DataFrame, wb_pdf, record_id: str) -> Optional[Dict]:
    """پردازش PDF (رفع off-by-one و محدود کردن partial به شماره سند)"""
    try:
        # نرمال‌سازی: فقط رقم‌ها
        document_number = safe_get_value(record, "document_number", "", str)
        doc_digits = re.sub(r'\D+', '', document_number or "")
        price_digits = re.sub(r'\D+', '', str(int(round(price))))  # 156548680

        if not doc_digits:
            app.logger.info(f"Record {record_id}: empty document_number after normalize.")
            return None

        ws_pdf = wb_pdf.active if wb_pdf else None

        for idx, row in pdf_data.iterrows():
            try:
                # رشته‌ی یکپارچه از سلول‌های همان ردیف
                row_cells = [str(c) if pd.notnull(c) else "" for c in row]
                joined_text = " ".join(x.replace("\n", " ") for x in row_cells)
                joined_digits = re.sub(r'\D+', '', joined_text)

                found_doc = doc_digits in joined_digits
                found_price = price_digits in joined_digits if price_digits else False

                # توجه: چون header=False است، ردیف دیتافریم 0 باید روی ردیف 1 اکسل بنشیند
                excel_row = idx + 1

                if found_doc and found_price:
                    if ws_pdf:
                        highlight_row(ws_pdf, excel_row, green_fill)
                    app.logger.info(
                        f"PDF FULL match at df_index={idx}, excel_row={excel_row}, doc={document_number}, price={price}"
                    )
                    return {"type": "full", "row": excel_row, "debug": [f"pdf_full idx={idx} row={excel_row}"]}

                # فقط اگر شماره سند پیدا شد، partial بده (برای جلوگیری از هایلایت مبلغ‌های تکراری)
                elif found_doc:
                    if ws_pdf:
                        highlight_row(ws_pdf, excel_row, red_fill)
                    app.logger.info(
                        f"PDF PARTIAL (doc only) at df_index={idx}, excel_row={excel_row}, doc={document_number}"
                    )
                    return {"type": "partial", "row": excel_row, "debug": [f"pdf_partial idx={idx} row={excel_row}"]}

                # اگر فقط مبلغ بود، رد شو تا ردیف اشتباه هایلایت نشه
            except Exception as e:
                app.logger.error(f"Row error in PDF (df_index {idx}): {str(e)}")
                continue

    except Exception as e:
        app.logger.error(f"Error in process_pdf: {str(e)}")

    return None