import openpyxl
from openpyxl.styles import PatternFill
import logging
import os
from typing import Optional, List, Dict, Any

# پیکربندی logger
logger = logging.getLogger(__name__)

# رنگ‌های پیش‌فرض
green_fill = PatternFill(start_color="00FF00", end_color="00FF00", fill_type="solid")
red_fill = PatternFill(start_color="FF0000", end_color="FF0000", fill_type="solid")
yellow_fill = PatternFill(start_color="FFFF00", end_color="FFFF00", fill_type="solid")

def highlight_row(ws, row_idx: int, color: PatternFill) -> bool:
    """
    رنگ‌آمیزی یک سطر در worksheet با مدیریت خطا
    
    Args:
        ws: Worksheet object
        row_idx: شماره سطر
        color: رنگ PatternFill
    
    Returns:
        bool: True در صورت موفقیت، False در صورت خطا
    """
    try:
        # بررسی معتبر بودن row_idx
        if row_idx < 1:
            logger.warning(f"Invalid row index: {row_idx}")
            return False
        
        # بررسی وجود worksheet
        if ws is None:
            logger.warning("Worksheet is None")
            return False
        
        # بررسی محدوده معتبر
        if row_idx > ws.max_row:
            logger.warning(f"Row index {row_idx} exceeds max row {ws.max_row}")
            return False
        
        # اعمال رنگ به تمام ستون‌های سطر
        for col in range(1, ws.max_column + 1):
            try:
                cell = ws.cell(row=row_idx, column=col)
                if cell:
                    cell.fill = color
            except Exception as e:
                logger.error(f"Error highlighting cell [{row_idx}, {col}]: {str(e)}")
                continue
        
        logger.info(f"Successfully highlighted row {row_idx}")
        return True
        
    except Exception as e:
        logger.error(f"Error highlighting row {row_idx}: {str(e)}")
        return False

def highlight_pdf_row(pdf_path: str, output_path: str, search_text: str, 
                     price: str, color_rgb: tuple) -> bool:
    """
    هایلایت کردن یک سطر در PDF با مدیریت خطا
    
    Args:
        pdf_path: مسیر فایل PDF ورودی
        output_path: مسیر فایل PDF خروجی
        search_text: متن جستجو
        price: قیمت برای جستجو
        color_rgb: رنگ RGB به صورت tuple
    
    Returns:
        bool: True در صورت موفقیت، False در صورت خطا
    """
    try:
        # بررسی وجود کتابخانه PyMuPDF
        try:
            import fitz  # PyMuPDF
        except ImportError:
            logger.error("PyMuPDF library not installed. Install with: pip install PyMuPDF")
            return False
        
        # بررسی وجود فایل ورودی
        if not os.path.exists(pdf_path):
            logger.error(f"PDF file not found: {pdf_path}")
            return False
        
        # بررسی دسترسی خواندن
        if not os.access(pdf_path, os.R_OK):
            logger.error(f"No read permission for PDF: {pdf_path}")
            return False
        
        # ایجاد دایرکتوری خروجی در صورت عدم وجود
        output_dir = os.path.dirname(output_path)
        if output_dir and not os.path.exists(output_dir):
            os.makedirs(output_dir)
        
        doc = fitz.open(pdf_path)
        highlights_added = 0
        
        for page_num, page in enumerate(doc):
            try:
                words = page.get_text("words")
                row_coords = []
                
                for word_info in words:
                    try:
                        # پشتیبانی از فرمت‌های مختلف
                        if len(word_info) >= 5:
                            x0, y0, x1, y1, word = word_info[:5]
                        else:
                            logger.warning(f"Invalid word info format: {word_info}")
                            continue
                        
                        # جستجوی متن
                        if (search_text and search_text.lower() in word.lower()) or \
                           (price and price in word):
                            row_coords.append({
                                "x0": x0, "y0": y0, 
                                "x1": x1, "y1": y1, 
                                "text": word
                            })
                    except Exception as e:
                        logger.error(f"Error processing word on page {page_num}: {str(e)}")
                        continue
                
                # اعمال هایلایت اگر مورد یافت شد
                if row_coords:
                    try:
                        x0 = min(w["x0"] for w in row_coords)
                        y0 = min(w["y0"] for w in row_coords)
                        x1 = max(w["x1"] for w in row_coords)
                        y1 = max(w["y1"] for w in row_coords)
                        
                        rect = fitz.Rect(x0, y0, x1, y1)
                        annot = page.add_rect_annot(rect)
                        
                        # تنظیم رنگ
                        if isinstance(color_rgb, tuple) and len(color_rgb) == 3:
                            # نرمال‌سازی رنگ به بازه 0-1
                            normalized_color = tuple(c/255 if c > 1 else c for c in color_rgb)
                            annot.set_colors(stroke=normalized_color)
                        else:
                            annot.set_colors(stroke=(1, 0, 0))  # قرمز پیش‌فرض
                        
                        annot.update()
                        highlights_added += 1
                        
                    except Exception as e:
                        logger.error(f"Error adding annotation on page {page_num}: {str(e)}")
                        
            except Exception as e:
                logger.error(f"Error processing page {page_num}: {str(e)}")
                continue
        
        # ذخیره فایل
        doc.save(output_path)
        doc.close()
        
        logger.info(f"PDF processed: {highlights_added} highlights added, saved to {output_path}")
        return True
        
    except Exception as e:
        logger.error(f"Error processing PDF: {str(e)}")
        return False

def read_excel_smart_header(file_path: str, required_columns: List[str], 
                           max_scan_rows: int = 10) -> Optional[Any]:
    """
    خواندن هوشمند فایل اکسل با شناسایی خودکار سطر هدر
    
    Args:
        file_path: مسیر فایل اکسل
        required_columns: لیست ستون‌های مورد نیاز
        max_scan_rows: حداکثر تعداد سطرهای اسکن برای یافتن هدر
    
    Returns:
        DataFrame یا None در صورت خطا
    """
    try:
        # بررسی وجود pandas
        try:
            import pandas as pd
        except ImportError:
            logger.error("pandas library not installed")
            return None
        
        # بررسی وجود فایل
        if not os.path.exists(file_path):
            logger.error(f"File not found: {file_path}")
            return None
        
        # بررسی لیست ستون‌های مورد نیاز
        if not required_columns:
            logger.warning("No required columns specified, reading with default header")
            return pd.read_excel(file_path)
        
        # جستجوی سطر هدر
        for header_row in range(max_scan_rows):
            try:
                df = pd.read_excel(file_path, header=header_row)
                
                # نرمال‌سازی نام ستون‌ها
                cols = [str(col).strip().lower() for col in df.columns]
                required_cols_normalized = [str(col).strip().lower() for col in required_columns]
                
                # بررسی وجود تمام ستون‌های مورد نیاز
                if all(col in cols for col in required_cols_normalized):
                    logger.info(f"Header found at row {header_row} in {file_path}")
                    return df
                    
            except Exception as e:
                logger.debug(f"Failed to read with header at row {header_row}: {str(e)}")
                continue
        
        # اگر هدر پیدا نشد
        logger.warning(f"Required columns not found in first {max_scan_rows} rows: {required_columns}")
        
        # تلاش برای خواندن بدون هدر
        try:
            df = pd.read_excel(file_path, header=None)
            logger.info(f"Read {file_path} without header, shape: {df.shape}")
            return df
        except Exception as e:
            logger.error(f"Failed to read file without header: {str(e)}")
            return None
            
    except Exception as e:
        logger.error(f"Error reading Excel file {file_path}: {str(e)}")
        return None

def validate_excel_structure(workbook) -> Dict[str, Any]:
    """
    بررسی ساختار فایل اکسل
    
    Args:
        workbook: Openpyxl workbook object
    
    Returns:
        دیکشنری حاوی اطلاعات ساختار
    """
    try:
        if workbook is None:
            return {"valid": False, "error": "Workbook is None"}
        
        info = {
            "valid": True,
            "sheet_count": len(workbook.sheetnames),
            "sheets": []
        }
        
        for sheet_name in workbook.sheetnames:
            try:
                ws = workbook[sheet_name]
                sheet_info = {
                    "name": sheet_name,
                    "max_row": ws.max_row,
                    "max_column": ws.max_column,
                    "has_data": ws.max_row > 0 and ws.max_column > 0
                }
                info["sheets"].append(sheet_info)
            except Exception as e:
                logger.error(f"Error analyzing sheet {sheet_name}: {str(e)}")
                sheet_info = {
                    "name": sheet_name,
                    "error": str(e)
                }
                info["sheets"].append(sheet_info)
        
        return info
        
    except Exception as e:
        logger.error(f"Error validating Excel structure: {str(e)}")
        return {"valid": False, "error": str(e)}

def safe_cell_value(cell) -> Optional[Any]:
    """
    خواندن امن مقدار سلول
    
    Args:
        cell: سلول اکسل
    
    Returns:
        مقدار سلول یا None
    """
    try:
        if cell is None:
            return None
        return cell.value
    except Exception as e:
        logger.error(f"Error reading cell value: {str(e)}")
        return None

def create_backup(file_path: str) -> Optional[str]:
    """
    ایجاد نسخه پشتیبان از فایل
    
    Args:
        file_path: مسیر فایل اصلی
    
    Returns:
        مسیر فایل پشتیبان یا None در صورت خطا
    """
    try:
        if not os.path.exists(file_path):
            logger.warning(f"File not found for backup: {file_path}")
            return None
        
        from datetime import datetime
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        
        base_name = os.path.basename(file_path)
        name_parts = os.path.splitext(base_name)
        backup_name = f"{name_parts[0]}_backup_{timestamp}{name_parts[1]}"
        
        backup_dir = os.path.join(os.path.dirname(file_path), "backups")
        if not os.path.exists(backup_dir):
            os.makedirs(backup_dir)
        
        backup_path = os.path.join(backup_dir, backup_name)
        
        import shutil
        shutil.copy2(file_path, backup_path)
        
        logger.info(f"Backup created: {backup_path}")
        return backup_path
        
    except Exception as e:
        logger.error(f"Error creating backup: {str(e)}")
        return None