import os
from pathlib import Path
from dotenv import load_dotenv

# Load environment variables
env_path = Path('.') / '.env'
load_dotenv(dotenv_path=env_path)

class Config:
    """تنظیمات پایه برنامه"""
    
    # Flask settings
    SECRET_KEY = os.getenv('SECRET_KEY', 'dev-secret-key-change-in-production')
    DEBUG = False
    TESTING = False
    
    # File upload settings
    MAX_CONTENT_LENGTH = int(os.getenv('MAX_CONTENT_LENGTH', 16 * 1024 * 1024))  # 16MB
    UPLOAD_FOLDER = os.getenv('UPLOAD_FOLDER', 'uploads')
    ALLOWED_EXTENSIONS = {'xlsx', 'xls', 'pdf', 'csv'}
    
    # API settings
    API_TIMEOUT = int(os.getenv('API_TIMEOUT', 30))
    MAX_RETRIES = int(os.getenv('MAX_RETRIES', 3))
    
    # CRM settings
    CRM_API_URL = os.getenv('CRM_API_URL', None)
    CRM_API_KEY = os.getenv('CRM_API_KEY', None)
    
    # CORS settings
    CORS_ORIGINS = os.getenv('ALLOWED_ORIGINS', '*').split(',')
    
    # Database settings (if needed in future)
    DATABASE_URI = os.getenv('DATABASE_URI', 'sqlite:///app.db')
    
    # Logging settings
    LOG_LEVEL = os.getenv('LOG_LEVEL', 'INFO')
    LOG_FORMAT = '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    LOG_FILE_MAX_BYTES = int(os.getenv('LOG_FILE_MAX_BYTES', 10 * 1024 * 1024))  # 10MB
    LOG_FILE_BACKUP_COUNT = int(os.getenv('LOG_FILE_BACKUP_COUNT', 5))
    
    # Processing settings
    MAX_WORKERS = int(os.getenv('MAX_WORKERS', 4))
    BATCH_SIZE = int(os.getenv('BATCH_SIZE', 100))
    
    # Cache settings
    CACHE_TYPE = os.getenv('CACHE_TYPE', 'simple')
    CACHE_DEFAULT_TIMEOUT = int(os.getenv('CACHE_DEFAULT_TIMEOUT', 300))
    
    @staticmethod
    def init_app(app):
        """Initialize application with config"""
        pass

class DevelopmentConfig(Config):
    """تنظیمات محیط توسعه"""
    DEBUG = True
    DEVELOPMENT = True
    
    @classmethod
    def init_app(cls, app):
        Config.init_app(app)
        # Development specific initialization
        import logging
        logging.basicConfig(level=logging.DEBUG)

class ProductionConfig(Config):
    """تنظیمات محیط Production"""
    DEBUG = False
    
    @classmethod
    def init_app(cls, app):
        Config.init_app(app)
        
        # Production specific initialization
        import logging
        from logging.handlers import SysLogHandler
        
        # Send logs to syslog in production
        if not app.debug and os.getenv('SYSLOG_ADDRESS'):
            syslog_handler = SysLogHandler(address=os.getenv('SYSLOG_ADDRESS'))
            syslog_handler.setLevel(logging.WARNING)
            app.logger.addHandler(syslog_handler)
        
        # Use secure headers
        @app.after_request
        def set_secure_headers(response):
            response.headers['X-Content-Type-Options'] = 'nosniff'
            response.headers['X-Frame-Options'] = 'DENY'
            response.headers['X-XSS-Protection'] = '1; mode=block'
            response.headers['Strict-Transport-Security'] = 'max-age=31536000; includeSubDomains'
            return response

class TestingConfig(Config):
    """تنظیمات محیط تست"""
    TESTING = True
    DEBUG = True
    
    # Use in-memory database for tests
    DATABASE_URI = 'sqlite:///:memory:'
    
    # Disable CSRF for testing
    WTF_CSRF_ENABLED = False
    
    @classmethod
    def init_app(cls, app):
        Config.init_app(app)

# Configuration dictionary
config = {
    'development': DevelopmentConfig,
    'production': ProductionConfig,
    'testing': TestingConfig,
    'default': DevelopmentConfig
}

def get_config(config_name=None):
    """Get configuration object based on environment"""
    if config_name is None:
        config_name = os.getenv('FLASK_ENV', 'development')
    return config.get(config_name, DevelopmentConfig)