"""
مدیریت متمرکز خطاها و استثناءها
"""

import logging
import traceback
from functools import wraps
from typing import Any, Callable, Dict, Optional, Tuple
from flask import jsonify, request

logger = logging.getLogger(__name__)

class AppError(Exception):
    """کلاس پایه برای خطاهای برنامه"""
    status_code = 500
    
    def __init__(self, message: str, status_code: Optional[int] = None, payload: Optional[Dict] = None):
        super().__init__()
        self.message = message
        if status_code is not None:
            self.status_code = status_code
        self.payload = payload
    
    def to_dict(self) -> Dict:
        rv = dict(self.payload or ())
        rv['message'] = self.message
        rv['status'] = 'error'
        return rv

class ValidationError(AppError):
    """خطای اعتبارسنجی داده‌ها"""
    status_code = 400

class FileNotFoundError(AppError):
    """خطای عدم وجود فایل"""
    status_code = 404

class ProcessingError(AppError):
    """خطای پردازش"""
    status_code = 422

class ExternalAPIError(AppError):
    """خطای ارتباط با API خارجی"""
    status_code = 502

class RateLimitError(AppError):
    """خطای محدودیت نرخ درخواست"""
    status_code = 429

def handle_app_error(error: AppError) -> Tuple[Dict, int]:
    """مدیریت خطاهای برنامه"""
    response = error.to_dict()
    logger.error(f"Application error: {error.message}", extra={
        'status_code': error.status_code,
        'payload': error.payload,
        'request_path': request.path if request else None,
        'request_method': request.method if request else None
    })
    return jsonify(response), error.status_code

def handle_validation_error(error: ValidationError) -> Tuple[Dict, int]:
    """مدیریت خطاهای اعتبارسنجی"""
    return handle_app_error(error)

def handle_404(error) -> Tuple[Dict, int]:
    """مدیریت خطای 404"""
    return jsonify({
        'status': 'error',
        'message': 'Resource not found',
        'path': request.path
    }), 404

def handle_500(error) -> Tuple[Dict, int]:
    """مدیریت خطای 500"""
    logger.error(f"Internal server error: {str(error)}", exc_info=True)
    
    # در محیط development جزئیات بیشتری نمایش بده
    if request and request.environ.get('FLASK_ENV') == 'development':
        return jsonify({
            'status': 'error',
            'message': 'Internal server error',
            'details': str(error),
            'traceback': traceback.format_exc()
        }), 500
    
    return jsonify({
        'status': 'error',
        'message': 'An internal error occurred. Please try again later.'
    }), 500

def safe_execute(func: Callable, *args, **kwargs) -> Tuple[bool, Any, Optional[str]]:
    """
    اجرای ایمن یک تابع با مدیریت خطا
    
    Returns:
        Tuple of (success, result, error_message)
    """
    try:
        result = func(*args, **kwargs)
        return True, result, None
    except Exception as e:
        error_msg = f"Error in {func.__name__}: {str(e)}"
        logger.error(error_msg, exc_info=True)
        return False, None, error_msg

def with_error_handling(default_return=None, raise_on_error=False):
    """
    Decorator برای مدیریت خطاها در توابع
    
    Args:
        default_return: مقدار بازگشتی در صورت خطا
        raise_on_error: آیا خطا را raise کند یا خیر
    """
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                logger.error(f"Error in {func.__name__}: {str(e)}", exc_info=True)
                
                if raise_on_error:
                    raise
                
                if default_return is not None:
                    return default_return
                
                # برای Flask routes
                if request:
                    return jsonify({
                        'status': 'error',
                        'message': f'Error in {func.__name__}',
                        'details': str(e) if request.environ.get('FLASK_ENV') == 'development' else None
                    }), 500
                
                return None
        
        return wrapper
    return decorator

def validate_required_fields(data: Dict, required_fields: list) -> Tuple[bool, Optional[str]]:
    """
    بررسی وجود فیلدهای الزامی
    
    Args:
        data: دیکشنری داده‌ها
        required_fields: لیست فیلدهای الزامی
    
    Returns:
        Tuple of (is_valid, error_message)
    """
    if not data:
        return False, "No data provided"
    
    missing_fields = []
    for field in required_fields:
        if field not in data or data[field] is None or data[field] == "":
            missing_fields.append(field)
    
    if missing_fields:
        return False, f"Missing required fields: {', '.join(missing_fields)}"
    
    return True, None

def validate_file_upload(file) -> Tuple[bool, Optional[str]]:
    """
    بررسی فایل آپلود شده
    
    Args:
        file: فایل آپلود شده
    
    Returns:
        Tuple of (is_valid, error_message)
    """
    if not file:
        return False, "No file provided"
    
    if file.filename == '':
        return False, "No file selected"
    
    # بررسی پسوند فایل
    allowed_extensions = {'xlsx', 'xls', 'pdf', 'csv'}
    file_ext = file.filename.rsplit('.', 1)[1].lower() if '.' in file.filename else ''
    
    if file_ext not in allowed_extensions:
        return False, f"File type not allowed. Allowed types: {', '.join(allowed_extensions)}"
    
    # بررسی حجم فایل (16MB)
    file.seek(0, 2)  # رفتن به انتهای فایل
    file_size = file.tell()
    file.seek(0)  # برگشت به ابتدای فایل
    
    max_size = 16 * 1024 * 1024  # 16MB
    if file_size > max_size:
        return False, f"File size exceeds maximum limit ({max_size // (1024*1024)}MB)"
    
    return True, None

def register_error_handlers(app):
    """
    ثبت error handler ها در Flask app
    
    Args:
        app: Flask application instance
    """
    app.errorhandler(AppError)(handle_app_error)
    app.errorhandler(ValidationError)(handle_validation_error) 
    app.errorhandler(404)(handle_404)
    app.errorhandler(500)(handle_500)
    
    @app.errorhandler(Exception)
    def handle_unexpected_error(error):
        logger.error(f"Unexpected error: {str(error)}", exc_info=True)
        
        if app.debug:
            return jsonify({
                'status': 'error',
                'message': 'Unexpected error occurred',
                'error_type': type(error).__name__,
                'details': str(error),
                'traceback': traceback.format_exc()
            }), 500
        
        return jsonify({
            'status': 'error',
            'message': 'An unexpected error occurred'
        }), 500
    
    logger.info("Error handlers registered successfully")